/* cio.c   by Michael Thorpe   2002-03-31 */

#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/socket.h>
#include "ftpd.h"
#include "telnet.h"

#define MAX_LINE 8192

#define CONNSTATE_NORM 0
#define CONNSTATE_IAC  1
#define CONNSTATE_DO   2
#define CONNSTATE_WILL 3

struct generic_message {
   int status;
   char *message;
};

static struct generic_message generic_messages[]={
   {555,"Requested action not taken: type or stru mismatch."}, /* RFC1123 */
   {554,"Requested action not taken: invalid REST parameter."}, /* RFC1123 */
   {553,"Requested action not taken. File name not allowed."},
   {552,"Requested file action aborted. Exceeded storage allocation."},
   {551,"Requested action aborted: page type unknown."},
   {550,"Requested action not taken. File unavailable."},
   {532,"Need account for storing files."},
   {530,"Not logged in."},
   {504,"Command not implemented for that parameter."},
   {503,"Bad sequence of commands."},
   {502,"Command not implemented."},
   {501,"Syntax error in parameters or arguments."},
   {500,"Syntax error, command unrecognized."},
   {452,"Requested action not taken. Insufficient storage space in system."},
   {451,"Requested action aborted: local error in processing."},
   {450,"Requested file action not taken. File unavailable."},
   {426,"Connection closed; transfer aborted."},
   {425,"Can't open data connection."},
   {421,"Service not available, closing control connection."},
   {350,"Requested file action pending further information."},
   {332,"Need account for login."},
   {331,"User name okay, need password."},
/* {257,""PATHNAME" created."}, <special string> */
   {250,"Requested file action okay, completed."},
   {230,"User logged in, proceed."},
/* {227,"Entering Passive Mode (h1,h2,h3,h4,p1,p2)."}, <special string> */
   {226,"Closing data connection. Requested file action successful."},
   {225,"Data connection open; no transfer in progress."},
   {221,"Service closing control connection."},
   {220,"Service ready for new user."},
/* {215,"NAME system type."}, <special string> */
/* {214,"Help message."}, <special string> */
/* {213,"File status."}, <special string> */
/* {212,"Directory status."}, <special string> */
/* {211,"System status, or system help reply."}, <special string> */
   {202,"Command not implemented, superfluous at this site."},
   {200,"Command okay."},
   {150,"File status okay; about to open data connection."},
   {125,"Data connection already open; transfer starting."},
/* {120,"Service ready in nnn minutes."}, <special string> */
/* {110,"Restart marker reply."}, <special string> */
   {0,0}
};

int cwrite(conn *c,char *s,size_t l) {
   size_t i;
   char *tmp;

   if(MAX_OUT_BUF<l+c->coutbuflen)
      return(1);
   if(!c->coutbuflen) {
tryagain:
      i=write(c->fdc,s,l);
      if(i==-1) {
         if(errno==EINTR)
            goto tryagain;
         if(errno != EAGAIN)
            return(1);
      } else if(i==l) {
         return(0);
      } else {
         s+=i;
         l-=i;
      }
   }
   tmp=(char *)realloc(c->coutbuf,l+c->coutbuflen);
   if(!tmp) {
      log(c,LOG_FAIL,"Couldn't allocate memory!\n");
      return(1);
   }
   c->coutbuf=tmp;
   memcpy(c->coutbuf+c->coutbuflen,s,l);
   c->coutbuflen+=l;
   return(0);
}

int putline(conn *c,unsigned int status,char *message) {
   int i;
   char *s,*t;

   if(!message) {
      for(i=0;status<generic_messages[i].status;i++)
         ;
      if(generic_messages[i].status==status)
         message=generic_messages[i].message;
      if(!message) {
         log(c,LOG_FAIL,"Unimplemented error code!");
         putline(c,421,0);
         return(-1);
      }
   }
   while((s=strchr(message,'\n'))) {
      i=s-message;
      if(i && message[i-1]=='\r')
         i--;
      s++;
      t=(char *)malloc(i+6);
      if(!t)
         return(1);
      sprintf(t,"%03u-%.*s",status,i,message);
      log(c,LOG_REPLY,t);
      t[i+4]='\r';
      t[i+5]='\n';
      i=cwrite(c,t,i+6);
      free(t);
      if(i)
         return(i);
      message=s;
   }
   i=strlen(message);
   t=(char *)malloc(i+6);
   if(!t)
      return(1);
   sprintf(t,"%03u %s",status,message);
   log(c,LOG_REPLY,t);
   t[i+4]='\r';
   t[i+5]='\n';
   i=cwrite(c,t,i+6);
   free(t);
   return(i);
}

int cin(conn *c) {
   char *tmp;
   size_t i,j;

   if(c->cinbuflen>=MAX_IN_BUF)
      return(1);
   tmp=(char *)realloc(c->cinbuf,MAX_IN_BUF);
   if(!tmp) {
      log(c,LOG_FAIL,"Couldn't allocate memory!\n");
      return(1);
   }
   c->cinbuf=tmp;
   i=read(c->fdc,c->cinbuf+c->cinbuflen,MAX_IN_BUF-c->cinbuflen);
   if(i==-1)
      return(1);
   if(i==0)
      return(1);
   j=c->cinbuflen;
   c->cinbuflen+=i;
   tmp=(char *)realloc(c->cinbuf,c->cinbuflen);
   if(tmp)
      c->cinbuf=tmp;
   for(i=j;j<c->cinbuflen;j++) {
      switch(c->cinstate) {
      case CONNSTATE_NORM:
         if(c->cinbuf[j]==TELNET_IAC)
            c->cinstate=CONNSTATE_IAC;
         else {
            if(c->cinbuf[j]=='\r' || c->cinbuf[j]=='\n') {
               c->cinbuf[i]='\0';
               if(++j<c->cinbuflen && (c->cinbuf[j]=='\r' || c->cinbuf[j]=='\n'))
                  ++j;
               if(do_cmd(c,c->cinbuf)) {
                  log(c,LOG_FAIL,"Command returned non-zero!");
                  return(1);
               }
               c->cinbuflen-=j;
               if(c->cinbuflen)
                  memmove(c->cinbuf,&c->cinbuf[j],c->cinbuflen);
               i=0;
               j=-1;
            } else {
               if(i != j)
                  c->cinbuf[i]=c->cinbuf[j];
               i++;
            }
         }
         break;
      case CONNSTATE_IAC:
         if(c->cinbuf[j]==TELNET_DO)
            c->cinstate=CONNSTATE_DO;
         else if(c->cinbuf[j]==TELNET_WILL)
            c->cinstate=CONNSTATE_WILL;
         else {
            c->cinstate=CONNSTATE_NORM;
            if(c->cinbuf[j]==TELNET_EC && j<c->cinbuflen+1)
               j++;
         }
         break;
      case CONNSTATE_DO:
      case CONNSTATE_WILL:
         c->telnetcrap[0]=TELNET_IAC;
         c->telnetcrap[1]=(c->cinstate==CONNSTATE_DO)?TELNET_WONT:TELNET_DONT;
         c->telnetcrap[2]=c->cinbuf[j];
         c->cinstate=CONNSTATE_NORM;
         cwrite(c,c->telnetcrap,3);
         break;
      }
   }
   return(0);
}

int cout(conn *c) {
   size_t i;
   char *tmp;

   if(!c->coutbuflen)
      return(0);
tryagain:
   i=write(c->fdc,c->coutbuf,c->coutbuflen);
   if(i<=0) {
      if(i==-1 && errno==EINTR)
         goto tryagain;
      if(i==-1 && errno != EAGAIN)
         return(1);
      return(0);
   } else if(i<c->coutbuflen) {
      memmove(c->coutbuf,c->coutbuf+i,c->coutbuflen-i);
      tmp=(char *)realloc(c->coutbuf,c->coutbuflen-i);
      if(tmp)
         c->coutbuf=tmp;
      c->coutbuflen-=i;
   } else {
      free(c->coutbuf);
      c->coutbuf=0;
      c->coutbuflen=0;
   }
   return(0);
}

