/* net.c   by Michael Thorpe   2000-09-03 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#if USE_PROTOENT
#include <netdb.h>
#endif
#if !ALLOW_GLOBBING
#include <dirent.h>
#endif
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include "ftpd.h"

conn *baseconn=0;
listener *baselistener=0;

static int parsebyte(const char *s,const char **post,unsigned char *r) {
   unsigned char c=0,d;

   while(isdigit(*s)) {
      d=10*c+*s-'0';
      if(d<c)
         return(1);
      c=d;
      s++;
   }
   if(post)
      *post=s;
   *r=c;
   return(0);
}

int parseaddr(const char *s,const char **post,unsigned char a[4]) {
   int i;
   unsigned char b[4];

   for(i=0;i<4;i++) {
      if(parsebyte(s,&s,&b[i]))
         return(1);
      if(*s != ',' && *s != '.' && *s != '\0')
         return(1);
      if(*s)
         s++;
   }
   if(!b[0] && !b[1] && !b[2] && !b[3])
      return(1);
   if(post)
      *post=s;
   if(a)
      for(i=0;i<4;i++)
         a[i]=b[i];
   return(0);
}

int parseport(const char *s,const char **post,unsigned short *port) {
   unsigned char a[2];

   if(parsebyte(s,&s,a))
      return(1);
   if(*s != ',' && *s != '.')
      return(1);
   s++;
   if(parsebyte(s,&s,a+1))
      return(1);
   if(*s)
      return(1);
   if(!a[0] && !a[1])
      return(1);
   if(post)
      *post=s;
   *port=(a[0]<<8)+a[1];
   return(0);
}

int bindaddrport(const char *addr,unsigned short port) {
   int f;
   struct sockaddr_in sa;
#if USE_PROTOENT
   struct protoent *p;
#endif

   memset(&sa,0,sizeof(sa));
   sa.sin_family=AF_INET;
   if(addr) {
      if(!inet_aton(addr,&sa.sin_addr))
         return(-1);
   } else {
      sa.sin_addr.s_addr=INADDR_ANY;
   }
   sa.sin_port=htons(port);
#if USE_PROTOENT
   p=getprotobyname("tcp");
   if(!p)
      return(-1);
   f=socket(AF_INET,SOCK_STREAM,p->p_proto);
   endprotoent();
#else
   f=socket(AF_INET,SOCK_STREAM,0);
#endif
   if(f==-1)
      return(-1);
   if(set_sockopts(f)) {
      close(f);
      return(-1);
   }
   if(port && port<1024)
      if(dropuserid())
         return(-1);
   if(bind(f,(struct sockaddr *)&sa,sizeof(sa))) {
      close(f);
      return(-1);
   }
   return(f);
}

conn *newconn(int fd,struct sockaddr_in *sar,listener *l) {
   conn *c;
   struct sockaddr_in sa;
   int i;

   c=(conn *)malloc(sizeof(conn));
   if(!c) {
      log(c,LOG_FAIL,"Couldn't allocate memory");
      return(0);
   }
   memset(c,0,sizeof(*c));
   c->uid=BASE_UID;
   c->gid=BASE_GID;
   c->fdc=fd;
   c->fdd=-1;
   c->dstate=DSTATE_ACTIVE;
   unsettimeout(c);
   c->mode='S';
   c->type='A';
   c->subtype='N';
   c->structure='F';
   if(!sar) {
      i=sizeof(sa);
      if(getpeername(fd,(struct sockaddr *)&sa,&i)) {
         log(c,LOG_FAIL,"getpeername() failed");
         goto fail;
      }
      sar=&sa;
   }
   c->raddr=strdup(inet_ntoa(sar->sin_addr));
   if(!c->raddr) {
      log(c,LOG_FAIL,"Couldn't allocate memory");
      goto fail;
   }
   c->rport=ntohs(sar->sin_port);
   c->drport=ntohs(sar->sin_port);
   if(l && l->addr) {
      c->laddr=strdup(l->addr);
      c->lport=l->cport;
   } else {
      i=sizeof(sa);
      if(getsockname(fd,(struct sockaddr *)&sa,&i)) {
         log(c,LOG_FAIL,"getsockname() failed");
         goto fail;
      }
      c->laddr=strdup(inet_ntoa(sa.sin_addr));
      c->lport=ntohs(sa.sin_port);
   }
   c->dlport=c->lport-1;
   if(!c->laddr) {
      log(c,LOG_FAIL,"Couldn't allocate memory");
      goto fail;
   }
   c->next=baseconn;
   if(baseconn)
      baseconn->prev=c;
   baseconn=c;
   if(log(c,LOG_CONNECT,c->raddr))
      goto fail;
   return(c);
fail:
   if(c->laddr)
      free(c->laddr);
   if(c->raddr)
      free(c->raddr);
   free(c);
   return(0);
}

void delconn(conn *c) {
   log(c,LOG_DISCONNECT,c->raddr);
   if(c->prev)
      c->prev->next=c->next;
   else
      baseconn=c->next;
   if(c->next)
      c->next->prev=c->prev;
   shutdown(c->fdc,2);
   close(c->fdc);
   if(c->fdd != -1)
      close(c->fdd);
#if !MODE_PERCONN
   if(c->cwd)
      free(c->cwd);
#endif
   if(c->cinbuf)
      free(c->cinbuf);
   if(c->coutbuf)
      free(c->coutbuf);
   if(c->doutbuf)
      free(c->doutbuf);
#if !ALLOW_GLOBBING
   if(c->dir)
      closedir(c->dir);
#endif
   if(c->laddr)
      free(c->laddr);
   if(c->raddr)
      free(c->raddr);
   if(c->user)
      free(c->user);
   if(c->username)
      free(c->username);
   if(c->rnfr)
      free(c->rnfr);
   free(c);
   dropuserid();
}

int newlistener(const char *addr,unsigned short port) {
   listener *l;
   char tmp[22];

   for(l=baselistener;l;l=l->next)
      if(!strcmp(addr,l->addr) && port==l->cport)
         return(1);
   if(dropuserid())
      return(1);
   snprintf(tmp,21,"%s:%hu",(addr?addr:"*"),port);
   tmp[21]='\0';
   l=(struct listener *)malloc(sizeof(struct listener));
   if(!l) {
      log(0,LOG_FAIL,"Couldn't allocate memory");
      return(1);
   }
   l->fdc=bindaddrport(addr,port);
   if(l->fdc==-1) {
      log2(0,LOG_FAIL,"Couldn't bind to control port for ",tmp);
      goto fail;
   }
   if(listen(l->fdc,5)) {
      log2(0,LOG_FAIL,"Couldn't listen on ",tmp);
      goto failclose;
   }
   if(log2(0,LOG_NOTICE,"Listening on ",tmp))
      goto failclose;
   if(addr) {
      l->addr=strdup(addr);
      if(!l->addr) {
         log(0,LOG_FAIL,"Couldn't allocate memory");
         goto failclose;
      }
   } else
      l->addr=0;
   l->cport=port;
   l->prev=0;
   l->next=baselistener;
   if(baselistener)
      baselistener->prev=l;
   baselistener=l;
   return(0);
failclose:
   close(l->fdc);
fail:
   free(l);
   return(1);
}

int dellistener(const char *addr,unsigned short port) {
   listener *l;
   char tmp[22];

   for(l=baselistener;l;l=l->next) {
      if(port != l->cport)
         continue;
      if(addr==l->addr)
         break;
      if(addr && l->addr && !strcmp(addr,l->addr))
         break;
   }
   if(!l)
      return(1);
   close(l->fdc);
   l->fdc=-1;
   if(l->prev)
      l->prev->next=l->next;
   else if(baselistener==l)
      baselistener=l->next;
   if(l->next)
      l->next->prev=l->prev;
   snprintf(tmp,21,"%s:%hu",(l->addr?l->addr:"*"),l->cport);
   tmp[21]='\0';
   log2(0,LOG_NOTICE,"No longer listening on ",tmp);
   if(l->addr)
      free(l->addr);
   free(l);
   return(0);
}

int laccept(listener *l) {
   int i,s;
   conn *c;
   struct sockaddr_in sa;

   i=sizeof(sa);
   s=accept(l->fdc,(struct sockaddr *)&sa,&i);
   if(s<0)
      return(1);
#if REDO_NONBLOCK
   if(set_nonblock(s)) {
      close(s);
      return(1);
   }
#endif
#if MODE_PERCONN
   i=fork();
   if(i != 0) {
      close(s);
      return(i<0);
   }
   curpid=getpid();
#endif
   log(0,LOG_NOTICE,"Starting (child)");
   c=newconn(s,&sa,l);
#if MODE_PERCONN
   while(baselistener)
      dellistener(baselistener->addr,baselistener->cport);
#endif
   if(!c)
      return(-1);
   if(putline(c,220,0))
      return(1);
   return(0);
}

int lerror(listener *l) {
   char tmp[22];

   snprintf(tmp,21,"%s:%hu",(l->addr?l->addr:"*"),l->cport);
   tmp[21]='\0';
   log2(0,LOG_FAIL,"Listener error on ",tmp);
   dellistener(l->addr,l->cport);
   return(1);
}

int cmd_pasv(conn *c,char *arg) {
   int i,s;
   unsigned short port;
   struct sockaddr_in sa;
   char buf[49];

   s=bindaddrport(c->laddr,0);
   if(s<0)
      return(1);
   i=sizeof(sa);
   if(getsockname(s,(struct sockaddr *)&sa,&i)) {
      log(c,LOG_FAIL,"getsockname() failed");
      close(s);
      return(1);
   }
   if(listen(s,1)) {
      log(c,LOG_FAIL,"Couldn't listen passively");
      close(s);
      return(1);
   }
   port=ntohs(sa.sin_port);
   c->dlport=port;
   c->fdd=s;
   c->dstate=DSTATE_PASSIVE;
   sprintf(buf,"Entering Passive Mode (%s,%u,%u).",c->laddr,port>>8,port&255);
   for(s=24;buf[s] != ',';s++)
      if(buf[s]=='.')
         buf[s++]=',';
   return(putline(c,227,buf));
}

